﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/accessanalyzer/AccessAnalyzer_EXPORTS.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/accessanalyzer/model/FindingStatus.h>
#include <aws/core/utils/memory/stl/AWSMap.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/accessanalyzer/model/ResourceType.h>
#include <aws/core/utils/DateTime.h>
#include <aws/accessanalyzer/model/FindingChangeType.h>
#include <aws/accessanalyzer/model/ResourceControlPolicyRestriction.h>
#include <aws/accessanalyzer/model/FindingSource.h>
#include <utility>

namespace Aws
{
namespace Utils
{
namespace Json
{
  class JsonValue;
  class JsonView;
} // namespace Json
} // namespace Utils
namespace AccessAnalyzer
{
namespace Model
{

  /**
   * <p>An access preview finding generated by the access preview.</p><p><h3>See
   * Also:</h3>   <a
   * href="http://docs.aws.amazon.com/goto/WebAPI/accessanalyzer-2019-11-01/AccessPreviewFinding">AWS
   * API Reference</a></p>
   */
  class AccessPreviewFinding
  {
  public:
    AWS_ACCESSANALYZER_API AccessPreviewFinding() = default;
    AWS_ACCESSANALYZER_API AccessPreviewFinding(Aws::Utils::Json::JsonView jsonValue);
    AWS_ACCESSANALYZER_API AccessPreviewFinding& operator=(Aws::Utils::Json::JsonView jsonValue);
    AWS_ACCESSANALYZER_API Aws::Utils::Json::JsonValue Jsonize() const;


    ///@{
    /**
     * <p>The ID of the access preview finding. This ID uniquely identifies the element
     * in the list of access preview findings and is not related to the finding ID in
     * Access Analyzer.</p>
     */
    inline const Aws::String& GetId() const { return m_id; }
    inline bool IdHasBeenSet() const { return m_idHasBeenSet; }
    template<typename IdT = Aws::String>
    void SetId(IdT&& value) { m_idHasBeenSet = true; m_id = std::forward<IdT>(value); }
    template<typename IdT = Aws::String>
    AccessPreviewFinding& WithId(IdT&& value) { SetId(std::forward<IdT>(value)); return *this;}
    ///@}

    ///@{
    /**
     * <p>The existing ID of the finding in IAM Access Analyzer, provided only for
     * existing findings.</p>
     */
    inline const Aws::String& GetExistingFindingId() const { return m_existingFindingId; }
    inline bool ExistingFindingIdHasBeenSet() const { return m_existingFindingIdHasBeenSet; }
    template<typename ExistingFindingIdT = Aws::String>
    void SetExistingFindingId(ExistingFindingIdT&& value) { m_existingFindingIdHasBeenSet = true; m_existingFindingId = std::forward<ExistingFindingIdT>(value); }
    template<typename ExistingFindingIdT = Aws::String>
    AccessPreviewFinding& WithExistingFindingId(ExistingFindingIdT&& value) { SetExistingFindingId(std::forward<ExistingFindingIdT>(value)); return *this;}
    ///@}

    ///@{
    /**
     * <p>The existing status of the finding, provided only for existing findings.</p>
     */
    inline FindingStatus GetExistingFindingStatus() const { return m_existingFindingStatus; }
    inline bool ExistingFindingStatusHasBeenSet() const { return m_existingFindingStatusHasBeenSet; }
    inline void SetExistingFindingStatus(FindingStatus value) { m_existingFindingStatusHasBeenSet = true; m_existingFindingStatus = value; }
    inline AccessPreviewFinding& WithExistingFindingStatus(FindingStatus value) { SetExistingFindingStatus(value); return *this;}
    ///@}

    ///@{
    /**
     * <p>The external principal that has access to a resource within the zone of
     * trust.</p>
     */
    inline const Aws::Map<Aws::String, Aws::String>& GetPrincipal() const { return m_principal; }
    inline bool PrincipalHasBeenSet() const { return m_principalHasBeenSet; }
    template<typename PrincipalT = Aws::Map<Aws::String, Aws::String>>
    void SetPrincipal(PrincipalT&& value) { m_principalHasBeenSet = true; m_principal = std::forward<PrincipalT>(value); }
    template<typename PrincipalT = Aws::Map<Aws::String, Aws::String>>
    AccessPreviewFinding& WithPrincipal(PrincipalT&& value) { SetPrincipal(std::forward<PrincipalT>(value)); return *this;}
    template<typename PrincipalKeyT = Aws::String, typename PrincipalValueT = Aws::String>
    AccessPreviewFinding& AddPrincipal(PrincipalKeyT&& key, PrincipalValueT&& value) {
      m_principalHasBeenSet = true; m_principal.emplace(std::forward<PrincipalKeyT>(key), std::forward<PrincipalValueT>(value)); return *this;
    }
    ///@}

    ///@{
    /**
     * <p>The action in the analyzed policy statement that an external principal has
     * permission to perform.</p>
     */
    inline const Aws::Vector<Aws::String>& GetAction() const { return m_action; }
    inline bool ActionHasBeenSet() const { return m_actionHasBeenSet; }
    template<typename ActionT = Aws::Vector<Aws::String>>
    void SetAction(ActionT&& value) { m_actionHasBeenSet = true; m_action = std::forward<ActionT>(value); }
    template<typename ActionT = Aws::Vector<Aws::String>>
    AccessPreviewFinding& WithAction(ActionT&& value) { SetAction(std::forward<ActionT>(value)); return *this;}
    template<typename ActionT = Aws::String>
    AccessPreviewFinding& AddAction(ActionT&& value) { m_actionHasBeenSet = true; m_action.emplace_back(std::forward<ActionT>(value)); return *this; }
    ///@}

    ///@{
    /**
     * <p>The condition in the analyzed policy statement that resulted in a
     * finding.</p>
     */
    inline const Aws::Map<Aws::String, Aws::String>& GetCondition() const { return m_condition; }
    inline bool ConditionHasBeenSet() const { return m_conditionHasBeenSet; }
    template<typename ConditionT = Aws::Map<Aws::String, Aws::String>>
    void SetCondition(ConditionT&& value) { m_conditionHasBeenSet = true; m_condition = std::forward<ConditionT>(value); }
    template<typename ConditionT = Aws::Map<Aws::String, Aws::String>>
    AccessPreviewFinding& WithCondition(ConditionT&& value) { SetCondition(std::forward<ConditionT>(value)); return *this;}
    template<typename ConditionKeyT = Aws::String, typename ConditionValueT = Aws::String>
    AccessPreviewFinding& AddCondition(ConditionKeyT&& key, ConditionValueT&& value) {
      m_conditionHasBeenSet = true; m_condition.emplace(std::forward<ConditionKeyT>(key), std::forward<ConditionValueT>(value)); return *this;
    }
    ///@}

    ///@{
    /**
     * <p>The resource that an external principal has access to. This is the resource
     * associated with the access preview.</p>
     */
    inline const Aws::String& GetResource() const { return m_resource; }
    inline bool ResourceHasBeenSet() const { return m_resourceHasBeenSet; }
    template<typename ResourceT = Aws::String>
    void SetResource(ResourceT&& value) { m_resourceHasBeenSet = true; m_resource = std::forward<ResourceT>(value); }
    template<typename ResourceT = Aws::String>
    AccessPreviewFinding& WithResource(ResourceT&& value) { SetResource(std::forward<ResourceT>(value)); return *this;}
    ///@}

    ///@{
    /**
     * <p>Indicates whether the policy that generated the finding allows public access
     * to the resource.</p>
     */
    inline bool GetIsPublic() const { return m_isPublic; }
    inline bool IsPublicHasBeenSet() const { return m_isPublicHasBeenSet; }
    inline void SetIsPublic(bool value) { m_isPublicHasBeenSet = true; m_isPublic = value; }
    inline AccessPreviewFinding& WithIsPublic(bool value) { SetIsPublic(value); return *this;}
    ///@}

    ///@{
    /**
     * <p>The type of the resource that can be accessed in the finding.</p>
     */
    inline ResourceType GetResourceType() const { return m_resourceType; }
    inline bool ResourceTypeHasBeenSet() const { return m_resourceTypeHasBeenSet; }
    inline void SetResourceType(ResourceType value) { m_resourceTypeHasBeenSet = true; m_resourceType = value; }
    inline AccessPreviewFinding& WithResourceType(ResourceType value) { SetResourceType(value); return *this;}
    ///@}

    ///@{
    /**
     * <p>The time at which the access preview finding was created.</p>
     */
    inline const Aws::Utils::DateTime& GetCreatedAt() const { return m_createdAt; }
    inline bool CreatedAtHasBeenSet() const { return m_createdAtHasBeenSet; }
    template<typename CreatedAtT = Aws::Utils::DateTime>
    void SetCreatedAt(CreatedAtT&& value) { m_createdAtHasBeenSet = true; m_createdAt = std::forward<CreatedAtT>(value); }
    template<typename CreatedAtT = Aws::Utils::DateTime>
    AccessPreviewFinding& WithCreatedAt(CreatedAtT&& value) { SetCreatedAt(std::forward<CreatedAtT>(value)); return *this;}
    ///@}

    ///@{
    /**
     * <p>Provides context on how the access preview finding compares to existing
     * access identified in IAM Access Analyzer.</p> <ul> <li> <p> <code>New</code> -
     * The finding is for newly-introduced access.</p> </li> <li> <p>
     * <code>Unchanged</code> - The preview finding is an existing finding that would
     * remain unchanged.</p> </li> <li> <p> <code>Changed</code> - The preview finding
     * is an existing finding with a change in status.</p> </li> </ul> <p>For example,
     * a <code>Changed</code> finding with preview status <code>Resolved</code> and
     * existing status <code>Active</code> indicates the existing <code>Active</code>
     * finding would become <code>Resolved</code> as a result of the proposed
     * permissions change.</p>
     */
    inline FindingChangeType GetChangeType() const { return m_changeType; }
    inline bool ChangeTypeHasBeenSet() const { return m_changeTypeHasBeenSet; }
    inline void SetChangeType(FindingChangeType value) { m_changeTypeHasBeenSet = true; m_changeType = value; }
    inline AccessPreviewFinding& WithChangeType(FindingChangeType value) { SetChangeType(value); return *this;}
    ///@}

    ///@{
    /**
     * <p>The preview status of the finding. This is what the status of the finding
     * would be after permissions deployment. For example, a <code>Changed</code>
     * finding with preview status <code>Resolved</code> and existing status
     * <code>Active</code> indicates the existing <code>Active</code> finding would
     * become <code>Resolved</code> as a result of the proposed permissions change.</p>
     */
    inline FindingStatus GetStatus() const { return m_status; }
    inline bool StatusHasBeenSet() const { return m_statusHasBeenSet; }
    inline void SetStatus(FindingStatus value) { m_statusHasBeenSet = true; m_status = value; }
    inline AccessPreviewFinding& WithStatus(FindingStatus value) { SetStatus(value); return *this;}
    ///@}

    ///@{
    /**
     * <p>The Amazon Web Services account ID that owns the resource. For most Amazon
     * Web Services resources, the owning account is the account in which the resource
     * was created.</p>
     */
    inline const Aws::String& GetResourceOwnerAccount() const { return m_resourceOwnerAccount; }
    inline bool ResourceOwnerAccountHasBeenSet() const { return m_resourceOwnerAccountHasBeenSet; }
    template<typename ResourceOwnerAccountT = Aws::String>
    void SetResourceOwnerAccount(ResourceOwnerAccountT&& value) { m_resourceOwnerAccountHasBeenSet = true; m_resourceOwnerAccount = std::forward<ResourceOwnerAccountT>(value); }
    template<typename ResourceOwnerAccountT = Aws::String>
    AccessPreviewFinding& WithResourceOwnerAccount(ResourceOwnerAccountT&& value) { SetResourceOwnerAccount(std::forward<ResourceOwnerAccountT>(value)); return *this;}
    ///@}

    ///@{
    /**
     * <p>An error.</p>
     */
    inline const Aws::String& GetError() const { return m_error; }
    inline bool ErrorHasBeenSet() const { return m_errorHasBeenSet; }
    template<typename ErrorT = Aws::String>
    void SetError(ErrorT&& value) { m_errorHasBeenSet = true; m_error = std::forward<ErrorT>(value); }
    template<typename ErrorT = Aws::String>
    AccessPreviewFinding& WithError(ErrorT&& value) { SetError(std::forward<ErrorT>(value)); return *this;}
    ///@}

    ///@{
    /**
     * <p>The sources of the finding. This indicates how the access that generated the
     * finding is granted. It is populated for Amazon S3 bucket findings.</p>
     */
    inline const Aws::Vector<FindingSource>& GetSources() const { return m_sources; }
    inline bool SourcesHasBeenSet() const { return m_sourcesHasBeenSet; }
    template<typename SourcesT = Aws::Vector<FindingSource>>
    void SetSources(SourcesT&& value) { m_sourcesHasBeenSet = true; m_sources = std::forward<SourcesT>(value); }
    template<typename SourcesT = Aws::Vector<FindingSource>>
    AccessPreviewFinding& WithSources(SourcesT&& value) { SetSources(std::forward<SourcesT>(value)); return *this;}
    template<typename SourcesT = FindingSource>
    AccessPreviewFinding& AddSources(SourcesT&& value) { m_sourcesHasBeenSet = true; m_sources.emplace_back(std::forward<SourcesT>(value)); return *this; }
    ///@}

    ///@{
    /**
     * <p>The type of restriction applied to the finding by the resource owner with an
     * Organizations resource control policy (RCP).</p>
     */
    inline ResourceControlPolicyRestriction GetResourceControlPolicyRestriction() const { return m_resourceControlPolicyRestriction; }
    inline bool ResourceControlPolicyRestrictionHasBeenSet() const { return m_resourceControlPolicyRestrictionHasBeenSet; }
    inline void SetResourceControlPolicyRestriction(ResourceControlPolicyRestriction value) { m_resourceControlPolicyRestrictionHasBeenSet = true; m_resourceControlPolicyRestriction = value; }
    inline AccessPreviewFinding& WithResourceControlPolicyRestriction(ResourceControlPolicyRestriction value) { SetResourceControlPolicyRestriction(value); return *this;}
    ///@}
  private:

    Aws::String m_id;
    bool m_idHasBeenSet = false;

    Aws::String m_existingFindingId;
    bool m_existingFindingIdHasBeenSet = false;

    FindingStatus m_existingFindingStatus{FindingStatus::NOT_SET};
    bool m_existingFindingStatusHasBeenSet = false;

    Aws::Map<Aws::String, Aws::String> m_principal;
    bool m_principalHasBeenSet = false;

    Aws::Vector<Aws::String> m_action;
    bool m_actionHasBeenSet = false;

    Aws::Map<Aws::String, Aws::String> m_condition;
    bool m_conditionHasBeenSet = false;

    Aws::String m_resource;
    bool m_resourceHasBeenSet = false;

    bool m_isPublic{false};
    bool m_isPublicHasBeenSet = false;

    ResourceType m_resourceType{ResourceType::NOT_SET};
    bool m_resourceTypeHasBeenSet = false;

    Aws::Utils::DateTime m_createdAt{};
    bool m_createdAtHasBeenSet = false;

    FindingChangeType m_changeType{FindingChangeType::NOT_SET};
    bool m_changeTypeHasBeenSet = false;

    FindingStatus m_status{FindingStatus::NOT_SET};
    bool m_statusHasBeenSet = false;

    Aws::String m_resourceOwnerAccount;
    bool m_resourceOwnerAccountHasBeenSet = false;

    Aws::String m_error;
    bool m_errorHasBeenSet = false;

    Aws::Vector<FindingSource> m_sources;
    bool m_sourcesHasBeenSet = false;

    ResourceControlPolicyRestriction m_resourceControlPolicyRestriction{ResourceControlPolicyRestriction::NOT_SET};
    bool m_resourceControlPolicyRestrictionHasBeenSet = false;
  };

} // namespace Model
} // namespace AccessAnalyzer
} // namespace Aws
